// $Id: WithArray.java,v 1.1 1999/09/16 12:31:08 oliva Exp $

/* Copyright 1999 Alexandre Oliva <oliva@lsd.ic.unicamp.br>
 *
 * This file is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package BR.unicamp.Guarana.SequentialComposerAlgorithms;

import BR.unicamp.Guarana.Message;
import BR.unicamp.Guarana.MetaObject;
import BR.unicamp.Guarana.Operation;
import BR.unicamp.Guarana.OperationFactory;
import BR.unicamp.Guarana.Result;
import java.util.Enumeration;

/** Implements algorithms for Composers like SequentialComposer, that
    sequentially request a fixed set of MetaObjects to handle
    Operations, Results and Messages, as well as reconfiguration,
    configuration, initialize and release requests.  It assumes the
    array of metaObjects given to each of its Operations is constant.

    <p>For each method of MetaObject, this class implements a method
    that takes the additional argument metaObjects, the array of
    MetaObjects to which it should delegate the requests.  Each such
    method invokes the corresponding method with the additional
    arguments begin, increment and range, implemented in the class
    WithArrayAndRange.  Begin and end are such that the full array is
    iterated on, and increment is -1 for handleResult and release, and
    +1 for all other methods.

    <p>All algorithms are final, to avoid exposing the metaObjects
    array of its main subclass, SequentialComposer.  This could have
    been avoided differently (with final implementations of these
    methods in class SequentialComposer itself), but it probably
    wasn't worth the trouble, since there's little point in
    specializing these methods.  They used to be static within
    SequentialComposer, after all.

    @see SequentialComposer.

    @since Guaran 1.7

    @author Alexandre Oliva
    @version $Revision: 1.1 $ */
public abstract class WithArray extends WithArrayAndRange {
  /** Invokes the corresponding method that takes a range.

      @see WithArrayAndRange#getMetaObjects  */
  final protected Enumeration
  getMetaObjects(final MetaObject[] metaObjects) {
    return getMetaObjects(metaObjects, 0, 1, metaObjects.length);
  }

  /** Invokes the corresponding method that takes a range.

      @see WithArrayAndRange#handleOperation
  */
  final protected Result
  handleOperation(final Operation operation,
		  final Object object,
		  final MetaObject[] metaObjects) {
    return handleOperation(operation, object,
			   metaObjects, 0, 1, metaObjects.length);
  }

  /** Invokes the corresponding method that takes a range.

      @see WithArrayAndRange#handleResult
  */
  final protected Result
  handleResult(Result res,
	       final Object object,
	       final MetaObject[] metaObjects) {
    return handleResult(res, object,
			metaObjects, metaObjects.length-1, -1, -1);
  }

  /** Invokes the corresponding method that takes a range.

      @see WithArrayAndRange#handleMessage
  */
  final protected void
  handleMessage(final Message message,
		final Object object,
		final MetaObject[] metaObjects) {
    handleMessage(message, object,
		  metaObjects, 0, 1, metaObjects.length);
  }

  /** Invokes the corresponding method that takes a range.

      @see WithArrayAndRange#configureArray
  */
  final protected MetaObject[]
  configureArray(final Object newObject,
		 final Object object,
		 final MetaObject[] metaObjects) {
    return configureArray(newObject, object,
			  metaObjects, 0, 1, metaObjects.length);
  }

  /** Invokes configureArray and composerForConfigure to determine the
      MetaObject to be returned.

      @see BR.unicamp.Guarana.SequentialComposerAlgorithms.WithArray#configure
      @see #composerForConfigure  */
  protected MetaObject
  configure(final Object newObject,
	    final Object object,
	    final MetaObject[] metaObjects) {
    return composerForConfigure(newObject, object,
				configureArray(newObject, object,
					       metaObjects));
  }

  /** Invokes the corresponding method that takes a range.

      @see WithArrayAndRange#reconfigureArray  */
  final protected MetaObject[]
  reconfigureArray(final Object object,
		   final MetaObject oldMetaObject,
		   final MetaObject newMetaObject,
		   final MetaObject[] metaObjects) {
    return reconfigureArray(object, oldMetaObject, newMetaObject,
			    metaObjects, 0, 1, metaObjects.length);
  }

  /** Invokes reconfigureArray and composerForReconfigure to determine
      the MetaObject to be returned.

      <p>It is worth noting that this implementation disregards
      requests to reconfigure itself, i.e., with <tt>oldMetaObject ==
      this</tt> or <tt>null</tt>, blindly passing them on to
      components.  Subclasses should take care of such
      reconfigurations, if appropriate.

      @see #reconfigureArray
      @see #composerForReconfigure  */
  protected synchronized MetaObject
  reconfigure(final Object object,
	      final MetaObject oldMetaObject,
	      final MetaObject newMetaObject,
	      final MetaObject[] metaObjects) {
    return composerForReconfigure(object,
				  oldMetaObject,
				  newMetaObject,
				  reconfigureArray(object,
						   oldMetaObject,
						   newMetaObject,
						   metaObjects));
  }

  /** Invokes the corresponding method that takes a range.

      @see WithArrayAndRange#initialize  */
  final protected void
  initialize(final OperationFactory factory,
	     final Object object,
	     final MetaObject[] metaObjects) {
    initialize(factory, object,
	       metaObjects, 0, 1, metaObjects.length);
  }

  /** Invokes the corresponding method that takes a range.

      @see WithArrayAndRange#release
  */
  final protected void
  release(final Object object,
	  final MetaObject[] metaObjects) {
    release(object,
	    metaObjects, metaObjects.length-1, -1, -1);
  }

  /** Invokes composerForArray.  This method is called from configure
      after the new array of metaObjects is determined.

      <p>Overriders of this method may invoke
      composerForArrayWithPolicies directly, with policies different
      from the default.

      @see #composerForArray
      @see #configure

      @param newObject the Object for which a new Composer should be
      provided.  Unused in the current implementation.  It may be
      useful for subclasses.

      @param object the Object whose meta-configuration is being
      requested to create the newObject's meta-configuration.  Unused
      in the current implementation.  It may be useful for
      subclasses.

      @param metaObjects the array of MetaObjects determined by the
      configure process (possibly null).

      @since Guaran 1.7  */
  protected MetaObject
  composerForConfigure(final Object newObject,
		       final Object object,
		       final MetaObject[] metaObjects) {
    return composerForArray(metaObjects);
  }

  /** Invokes composerForArray.  This method is called from configure
      after the new array of metaObjects is determined.

      <p>Overriders of this method may invoke
      composerForArrayWithPolicies directly, with policies different
      from the default.

      @see #composerForArray
      @see #reconfigure

      @param object the Object whose meta-configuration is being
      modified.  Unused in the current implementation.  It may be
      useful for subclasses.

      @param oldMetaObject the MetaObject to be replaced.  Unused in
      the current implementation.  It may be useful for subclasses.

      @param newMetaObject the MetaObject to replace it.  Unused in
      the current implementation.  It may be useful for subclasses.

      @param metaObjects the array of MetaObjects determined by the
      reconfigure process (possibly null).

      @since Guaran 1.7  */
  protected MetaObject
  composerForReconfigure(final Object object,
			 final MetaObject oldMetaObject,
			 final MetaObject newMetaObject,
			 final MetaObject[] metaObjects) {
    return composerForArray(metaObjects);
  }

  /** Invokes composerForArrayWithPolicies.  This method is called
      from composerForConfigure and composerForReconfigure.

      <p>Overriders of this method may invoke
      composerForArrayWithPolicies directly, with policies different
      from the default.

      @see #composerForArrayWithPolicies
      @see #composerForConfigure
      @see #composerForReconfigure

      @param metaObjects the array of MetaObjects determined by the
      configure or reconfigure process (possibly null).

      @return a MetaObject (typically a Composer) that delegates to
      the given metaObjects.

      @since Guaran 1.7  */
  protected MetaObject
  composerForArray(final MetaObject[] metaObjects) {
    return composerForArrayWithPolicies(metaObjects, true, true, true);
  }

  /** This method is invoked from composerForArray.  It uses
      newComposer to create the new Composer, when appropriate.

      @see #composerForArray
      @see #composerForConfigure
      @see #composerForReconfigure
      @see #newComposer

      @param metaObjects the array of MetaObjects determined by the
      configure or reconfigure process (possibly null).

      @param may_propagate_itself indicates whether this method should
      return <tt>this</tt> if the given array is exactly the one the
      Composer delegates to.  Enabling this is useful for
      reconfigurations that don't change any MetaObject, but, in some
      cases, it should be disabled for configurations.  This test is
      not implemented in this abstract class, since it can't tell what
      the original array is, but subclasses are encouraged to
      implement it.

      @param may_leave_when_empty indicates whether this method should
      return <tt>null</tt> when it gets a <tt>null</tt> array, or an
      array with length 0.  If the argument is <tt>false</tt>, a new
      Composer will be created with an array of length zero.

      @param may_leave_when_unitary indicates whether this method
      should return the only element of the array, when it finds the
      array is unitary.  If the argument is <tt>false</tt>, a new
      Composer will be created with the unitary array.

      @return a MetaObject (typically a Composer) that delegates to
      the given metaObjects.

      @since Guaran 1.7  */
  protected MetaObject
  composerForArrayWithPolicies(MetaObject[] metaObjects,
			       final boolean may_propagate_itself,
			       final boolean may_leave_when_empty,
			       final boolean may_leave_when_unitary) {
    if (metaObjects == null) {
      if (may_leave_when_empty)
	return null;
      else
	metaObjects = new MetaObject[0];
    }

    switch(metaObjects.length) {
    case 0:
      if (may_leave_when_empty)
	return null;
      break;

    case 1:
      if (may_leave_when_unitary)
	return metaObjects[0];
      break;
    }

    return newComposer(metaObjects);
  }

  /** Invoked by composerForArrayWithPolicies, it should create a new
      Composer that delegates to the given array of metaObjects.

      @param metaObjects the set of MetaObjects the new Composer
      should delegate to.

      @return the new Composer.

      @since Guaran 1.7  */
  protected abstract MetaObject
  newComposer(final MetaObject[] metaObjects);
}
