// $Id: gor1.cc,v 1.8 1998/11/29 18:38:45 islene Exp $

/* Copyright 1998 Alexandre Oliva <oliva@dcc.unicamp.br>, Islene Calciolari Garcia <islene@dcc.unicamp.br>
 *
 * This file is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 *  Topological-scan variation (GOR1): Consider that a depth-first
 *  search is used to compute the topological ordering.  When an arc
 *  (v,w) is examined by the depth-first search, the arc is first
 *  scanned in the shortest-path sense, that is, if d(v) + l(v,w) <
 *  d(w), d(w) is set to d(v) + l(v,w) and \pi(w) is set to v. (Note
 *  that this changes the admissible graph.)
 */

#include "spalgo.h"
#include "graph.h"
#include "ts1node.h"

#include <stack>

class topological_scan_var :
  public shortest_path_algorithm<topological_scan_var, graph<ts1node> > {
  /** Convenience type aliases.  */
  typedef shortest_path_algorithm<topological_scan_var, graph<ts1node> > inherited;
  typedef std::vector<node_t*> queue_t;

  /** Stacks of nodes to scan in the current pass and in the next pass.  */
  queue_t to_scan, to_scan_next;
  
  /** Inserts the node in the queue for the current pass.  */
  void push_this(node_t& node) {
    to_scan.push_back(&node);
  }

  /** Inserts the node in the queue for the next pass.  */
  void push(node_t& node) {
    to_scan_next.push_back(&node);
  }

  /** Returns true if there are not more nodes left to scan in the
      current pass.  */
  bool empty() const {
    return to_scan.empty();
  }

  /** Return true if the queue for the next pass is empty.  */
  bool next_empty() const {
    return to_scan_next.empty();
  }

  /** Removes a node from the top of the current-pass queue.  Make
      sure it is marked as not visited.  */
  node_t *pop() {
    node_t *node = to_scan.back();
    to_scan.pop_back();
    node->unvisit();
    return node;
  }

  /** Removes a node from the front of the next-pass queue.  */
  node_t *pop_next() {
    node_t *node = to_scan_next.back();
    to_scan_next.pop_back();
    return node;
  }

  /** Visit the given node.  Since the behavior for nodes removed from
      the next-pass queue is different from those that are just
      reached by them, the `from_next' flag is used to select the
      appropriate behavior.

      There are various admittedly strange decisions in this code,
      that we have taken for the sake of duplicating the algorithm of
      the original implementation by Cherkassky, Goldberg and Radzik.
      We'll try to clean up these apparent mistakes in another
      implementation. */
  void visit(node_t& node, bool from_next = false) {
    if (node.visit())
      return; // do not visit a node more than once in a pass

    /** Becomes true if we find any child in the admissible graph.  */
    bool found = false;

    adjlist_t &adjlist = node.adjlist(); // short-hand for adjacency list

    // In this algorithm, a node is always reached before it is
    // visited, so we need not test it.
    distance_t dist = node.distance();
    
    // Scan all arcs in the current node.
    for(arc_iterator arc_ptr = adjlist.begin(), end_arc = adjlist.end();
	arc_ptr != end_arc; ++arc_ptr) {
      arc_t& arc = *arc_ptr; // short-hand for the arc
      node_t& to = arc.to(); // the head of the arc
      if (to.test_relabel(arc)) { // if we have a shorter path
	found = true; // remember we must push the tail
	visit(to); // and visit the child
      }
    }
    
    if (found || !from_next) {
      push_this(node); // put it in the queue for this pass
      node.visited(); // mark it as visited

      // Note that we're ticking the counter when we push the node
      // into the queue, and then we tick it again after scanning it
      // in the main loop.  Yes, it is strange, but it's exactly what
      // the original implementation did.
      scan_tick(); 
    } else {
      // If we get here, no arc could be found in the admissible graph.
      scanned(node); // mark the node as scanned
      node.unvisit(); // and remove the visit mark
      // This is no good, since we might get back to this node.
      // Moreover, we're not counting this scan.  But then, the
      // original implementation isn't either.
    }
  }
      
public:
  /** Initialize the queue with the source node.  Note that it is the
      next pass queue that is initialized, so the algorithm starts
      with a topological scan.  Isn't this wasteful?  */
  topological_scan_var(graph_t& g) : inherited(g) {
    push(g.source()); // scan source first
  }

  /** Pick a node from the current-pass queue.  If the queue was
      empty, run the topological sorting heuristics to set up the next
      pass.  */
  node_t *select() {
    for(;;) {
      if (!empty())
	return pop(); // ok, we found a node
    
      if (next_empty())
	return 0; // job done, no more nodes to process
    
      do { // since next cannot empty here, we need not test it again
	node_t& node = *pop_next(); // pick a node from the next queue
	visit(node, true); // and visit it
      } while (!next_empty()); // until we have visited them all
    } // now restart
  }

  /** Return true iff the node was relabeled.  If it has become
      labeled, it is inserted in the queue.  */
  bool test_relabel(arc_t& arc) {
    bool was_labeled = arc.to().is_labeled();
    bool relabeled = inherited::test_relabel(arc);
    // was_labeled implies it was in the queue already
    if (relabeled && !was_labeled)
      push(arc.to()); // scan it in the next pass
    return relabeled;
  }
};

typedef topological_scan_var algorithm_t;

#include "main.h"
