// $Id: in2bin.cc,v 1.11 1998/11/29 05:36:44 oliva Exp $

/* Copyright 1998 Alexandre Oliva <oliva@dcc.unicamp.br>, Islene Calciolari Garcia <islene@dcc.unicamp.br>
 *
 * This file is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <string>

#include "graph.h"
#include "node.h"
#include "in2bin_node.h"
#include "timing.h"

/** Convenience type aliases.  */
typedef graph<in2bin_node> graph_t;
typedef graph_t::iterator node_iterator;
typedef graph_t::node_t node_t;
typedef node_t::node_id_t node_id_t;
typedef node_t::adjlist_t adjlist_t;
typedef node_t::distance_t distance_t;
typedef adjlist_t::arc_t arc_t;
typedef adjlist_t::iterator arc_iterator;

/** Distance type with `infinity' coded as uninitialized value.  */
class dist_init {
private:
  /** Indicates whether the value of dist is meaningful or not.  If
      false, take it as infinity.  */
  bool init;

  /** The value of the distance.  Only meaningful if init is true.  */
  distance_t dist;

public:
  /** Create infinite distance.  */
  dist_init() : init(false) {}
  /** Create finite distance.  */
  dist_init(distance_t d) : init(true), dist(d) {}

  /** Negate the distance value, unless it is infinite.  */
  dist_init operator-() const {
    if (init)
      return -dist;
    else
      return dist_init();
  }

  /** Add two distances, unless any or both of them are infinite.  */
  dist_init operator+(const dist_init& other) const {
    if (init && other.init)
      return dist + other.dist;
    else
      return dist_init();
  }

  /** If the second distance is infinite, return true.  Otherwise, if
      the first distance if infinite, return false.  Otherwise,
      compare the finite distances.  */
  friend bool operator<(const dist_init& a, const dist_init& b) {
    return (!b.init || (a.init && a.dist < b.dist));
  }

  /** Check whether a distance is finite.  */
  bool initialized() const { return init; }

  /** Obtain the value of the distance.  Should only be called if it
      is known to be finite.  */
  distance_t distance() const {
    // assert(init);
    return dist;
  }
};

/** Search and remove cycles of negative cost in the given graph.  */
void search_cycles(graph_t& graph, int &counter) {
  int node_count = graph.size();

  typedef vector<dist_init> line_t;
  typedef vector<line_t> matrix_t;

  // First, create a distance matrix
  matrix_t dist(node_count, line_t(node_count)); 
  
  // Initialize the diagonal with zeroes; all the rest is infinite.
  for(node_iterator i = graph.begin(), e = graph.end(); i != e; ++i) {
    node_id_t node = i->id()-1;
    dist[node][node] = 0;
  }

  // For each node
  for(node_iterator in = graph.begin(), e = graph.end(); in != e; ++in) {
    node_t &cur_node = *in; // the current node
    node_id_t cur_node_id = cur_node.id()-1; // and its subscript
    
    // For each delayed arc with head on the current node
    while(arc_t *arcp = cur_node.next_delayed_arc()) {
      node_id_t head = cur_node_id; // short-hand for current node id
      arc_t &arc = *arcp; // short-hand for arc
      distance_t length = arc.length(); // short-hand for arc length
      node_t &tail_node = arc.from(); // get tail node
      node_id_t tail = tail_node.id()-1; // and its subscript
      
      if (length < dist[tail][head]) {
	// if this arc shortens the shortest known path
	for(node_id_t i = 0; i < head; ++i) {
	  // compute the distance from i to head through the arc:
	  dist_init d = dist[i][tail] + length;
	  if (d.initialized() && d < dist[i][head])
	    // if the computed distance is shorter than the shortest
	    // previously known path from i to head, record it
	    dist[i][head] = d;
	}
      }
    }

    adjlist_t &adjlist = cur_node.adjlist(); // short-hand for adjacency list

    // For each arc with cur_node as tail
    for(arc_iterator ia = adjlist.begin(), e = adjlist.end(); ia != e; ++ia) {
      node_id_t tail = cur_node_id; // short-hand for current node id
      arc_t &arc = *ia; // short-hard for arc
      node_t &head_node = arc.to(); // get head node
      node_id_t head = head_node.id()-1; // and its subscript

      if (head > tail) { 
	// If the arc is a forward-arc, delay its processing until we
	// reach its head node.  Since we're scanning the nodes in
	// increasing order of id, we need not care about negative
	// cycles if the arc points to a node that was not visited
	// yet.  We can only get cycles if the arc points to a node
	// that we have already visited.  Moreover, with this
	// optimization, we need not maintain the distances to nodes
	// after tail correct before we reach it.
	head_node.delay(arc);
	continue;
      }

      // from now on we can assume head <= tail, i.e., we're dealing
      // with a back-arc

      distance_t length = arc.length(); // short-hand for arc length
      bool updated = false; // if we change it, it will become true

      if (length < dist[tail][head]) { // if we're shortening the path
	for(node_id_t i = 0; i <= tail; ++i) {
	  // compute the absolute weight of the other arcs of the
	  // cycle:
	  dist_init d = -(dist[i][tail] + dist[head][i]);
	  if (d.initialized() && length < d) { 
	    // length is considered shorter than an uninitialized d,
	    // but this does not form a negative cycle
	    // increase the length of the arc so that the weight of
	    // the cycle becomes zero
	    length = d.distance();
	    updated = true;
	  }
	}

	if (updated) {
#if VERBOSE_REMOVE
	  cerr << "cycle: tail: " << tail+1 << ", head: " << head+1
	       << ", length: " << length << ", was " << arc.length()
	       << endl;
#endif
	  ++counter; // tick the negative cycle counter
	  arc.set_length(length); // record the new length
	}

	// since we may have modified the length, test it again
	if (!updated || length < dist[tail][head]) {
	  // If we're shortening the path for this pair of nodes, we
	  // may well be able to shorten other paths involving this
	  // arc too.  So, let's look for paths such as:
	  // i -...-> tail -> head -...> j
	  // we need not test i > tail, since we haven't visited such
	  // nodes yet.
	  for(node_id_t i = 0; i <= tail; ++i) {
	    // compute the distance from i to head through the arc:
	    dist_init d = dist[i][tail] + length;
	    if (d.initialized() && d < dist[i][head]) {
	      // if the computed distance is shorter than the shortest
	      // previously known path from i to head, record it
	      dist[i][head] = d;
	      for(node_id_t j = 0; j <= tail; ++j) {
		// compute the distance from i to j through the arc
		dist_init d = dist[i][head] + dist[head][j];
		if (d.initialized() && d < dist[i][j])
		  // if it is shorter, record it
		  dist[i][j] = d;
	      }
	    }
	  }
	}
      }
    }
  }
}

int main(int argc, char *argv[]) {
  /** If ascii is true, the output will be in the DIMACS format.
      Otherwise, a non-cross-platform binary format will be used.

      If remove_cycles is true, the function above will be called to
      remove cycles of negative cost.  */
  bool ascii = false, remove_cycles = false;

  /** Process command-line arguments.  */
  for(int argn = 1; argn < argc; ++argn) {
    std::string arg = argv[argn];
    if (arg == "--ascii" || arg == "-a")
      ascii = true;
    else if (arg == "--binary" || arg == "-b")
      ascii = false;
    else if (arg == "--no-negative-cycles" || arg == "-n"
	     || arg == "--remove-negative-cycles" || arg == "-r")
      remove_cycles = true;
    else if (arg == "--help" || arg == "-h") {
      cout << "usage: [-a|-b] [-n|-r]" << endl;
      cout << "-a --ascii: write in ascii (DIMACS) format" << endl;
      cout << "-b --binary: write in binary non-portable format" << endl;
      cout << "-n --no-negative-cycles: increase cost of some arcs to remove negative cycles" << endl;
      cout << "-r --remove-negative-cycles: same as -n" << endl;
      exit(0);
    } else {
      cerr << "run with --help for help" << endl;
      exit(1);
    }
  }

  graph_t g(cin); // read the graph
  if (remove_cycles) {
    int counter = 0;
    cerr << "Removing negative cycles..." << endl;
    vtime_t start; getvtime(start); // start measuring the time
    search_cycles(g, counter); // run the negative cycle remover
    vtime_t end; getvtime(end); // done
    cerr << "Increased the weight of " << counter << " arcs" << endl;
    cerr << "Removal time: " << diffvtime(end, start) << endl;
  }
  if (ascii) {
    cout << "t " << g.get_title() << endl; // print out the title
    cout << "p sp " << g.size() << ' ' << g.arc_count() << endl; // the size
    cout << "n " << g.source().id() << endl; // and the source node
    for(graph_t::iterator i = g.begin(), e = g.end(); i != e; ++i) {
      node_t &node = *i; // iterate on all nodes
      adjlist_t &adjlist = node.adjlist();
      for(adjlist_t::iterator i = adjlist.begin(), e = adjlist.end();
	  i != e; ++i) {
	arc_t& arc = *i; // and all their arcs
	cout << "a " << arc.from().id() << ' ' << arc.to().id()
	     << ' ' << arc.length() << endl; // printing them
      }
    }
  } else {
    cout << "t " << g.get_title() << endl; // print out the title
    for(int i = 2 + g.get_title().length() + 1 + 1; i % sizeof(int); --i)
      cout << ' '; // int-boundary padding for better speed
    cout << 'b'; // start of non-portable binary sequence
    graph_t::header_t header = { g.size(), g.arc_count(), g.source().id() };
    cout.write(&header, sizeof(header)); // print out the header
    for(graph_t::iterator i = g.begin(), e = g.end(); i != e; ++i) {
      node_t &node = *i; // for each node
      adjlist_t &adjlist = node.adjlist();
      unsigned arcs = adjlist.size();
      cout.write(&arcs, sizeof(arcs)); // print out the number of arcs
      for(adjlist_t::iterator i = adjlist.begin(), e = adjlist.end();
	  i != e; ++i) {
	arc_t& arc = *i;
	graph_t::arc_data_t arc_data = { arc.to().id(), arc.length() };
	cout.write(&arc_data, sizeof(arc_data)); // and the adjacency list
      }
    }
  }
}
