#ifndef _node_h
#define _node_h

// $Id: node.h,v 1.7 1998/11/28 21:38:29 islene Exp $

/* Copyright 1998 Alexandre Oliva <oliva@dcc.unicamp.br>, Islene Calciolari Garcia <islene@dcc.unicamp.br>
 *
 * This file is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "distance.h"

/** Represents a node of a graph.  An implementation must provide the
    nested types node_t, adjlist_t, state_t, arc_t, distance_t and
    node_id_t.  A constructor that takes a node_id_t must be provided,
    as well as the accessor methods id(), adjlist(), parent_arc(),
    parent_node(), distance().  State-related methods are
    is_reached(), is_labeled().  Algorithm-related methods are
    set_source(), test_relabel(arc_t) and mark_scanned().  */
template <class node_T, class adjlist_T, class state_T,
  class distance_T = int, class node_id_T = size_t> class node {
 public:
  /** This is the actual type of the node, in case it has to be
      down-casted.  */
  typedef node_T node_t;

  /** This is the type of the adjacency list.  */
  typedef adjlist_T adjlist_t;
  /* This is the type of the arcs that link these nodes.  */
  typedef typename adjlist_t::arc_t arc_t;

  /** This is the class that implements the state machine of the node.
      The constructor of this class should mark the state of the node
      as unreached.  */
  typedef state_T state_t;

  /** This is the type used to denote distances between nodes.  Any
      signed integral type will do.  */
  typedef distance_T distance_t;

  /** This is the type used to identify nodes.  Any type will do, as
      long as it can be implicitly converted to integers in the range
      [1,n].  */
  typedef node_id_T node_id_t;
  
  /** Create a node given its identifier.  If you're using any type
      other than int for distances, look at distance.h to learn how to
      define the large() method.  */
  node(node_id_t id) : ID(id), parc(0),
  dist(distance_limits<distance_t>::large()) {}

 private:
  /** The identified of the node.  */
  node_id_t ID;
 public:
  /** Obtains the identified or the node.  */
  node_id_t id() const { return ID; }

 private:
  /** The adjacency list of the node.  */
  adjlist_t arcs;
 public:
  /** Obtains the adjacency list of the node.  */
  adjlist_t& adjlist() { return arcs; }

 protected:
  /** Holds the state of the node.  */
  state_t status;

 public:
  /** Returns true if the state is not unreached.  */
  bool is_reached() const { return status.is_reached(); }

  /** Returns true if the state is labeled.  */
  bool is_labeled() const { return status.is_labeled(); }

private:
  /** Marks the object as labeled.  */
  void mark_labeled() { status.mark_labeled(); }

 private:
  /** The incoming arc from the parent node, in the shortest path tree.  */
  arc_t *parc;
 public:
  /** Obtains the arc from the parent node, or NULL, if the node is
     the source or unreached.  */
  arc_t *parent_arc() const { return parc; }
  /** Obtains the parent node, or NULL, if the node is the source or
     unreached.  */
  node_t *parent_node() const { return parc ? &parc->from() : 0; }

 private:
  /** The distance from the source to this node, unless the node is
      unreached.  */
  distance_t dist;
 public:
  /** Obtains the distance from the source to this node.  Only
      meaningful if the node was reached already.  */
  distance_t distance() const { return dist; }
  
public:
  /** Mark this node as the source.  */
  void set_source() {
    mark_labeled();
    dist = 0;
    parc = 0;
  }

  /** Checks whether the given arc gives a shorter path that the
      shortest currently known path from the source to the head of the
      arc.  If it does, relabel the head node, update its distance and
      parent arc and return true.  Otherwise, return false.

      This method assumes the tail node was reached already, i.e., its
      distance field is meaningful.  */
  bool test_relabel(arc_t &arc) {
    distance_t distance = arc.from().distance() + arc.length();
    /** Compare distances first.  Even though the head distance may be
        uninitialized, it will be initialized in the common case, and,
        if it is not, we may even be lucky enough to not need to check
        whether the node was initialized already.  */
    if (distance < this->dist || !is_reached()) {
      mark_labeled();
      this->dist = distance;
      this->parc = &arc;
      return true;
    }
    return false;
  }

  /** Mark this node as scanned.  */
  void mark_scanned() { status.mark_scanned(); }
};

#endif
