#ifndef _spalgo_h
#define _spalgo_h

// $Id: spalgo.h,v 1.7 1998/11/29 06:22:36 oliva Exp $

/* Copyright 1998 Alexandre Oliva <oliva@dcc.unicamp.br>, Islene Calciolari Garcia <islene@dcc.unicamp.br>
 *
 * This file is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <iostream>

/** Generic base class for shortest path algorithms.  It makes various
    assumptions about the graph type, such as that it has a nested
    type named node_t, and node_t has nested types adjlist_t and
    distance_t, and adjlist_t has nested types iterator and arc_t.

    Upon construction, this class initializes the source node of the
    graph as the source.  */
template <class algorithm_T, class graph_T>
class shortest_path_algorithm {
 public:
  /** The derived algorithm type.  */
  typedef algorithm_T algorithm_t;
  
  /** The graph type.  */
  typedef graph_T graph_t;

  /** The node type.  */
  typedef typename graph_t::node_t node_t;

  /** The adjacency list type.  */
  typedef typename node_t::adjlist_t adjlist_t;

  /** An iterator type to iterate on arcs of an adjacency list.  */
  typedef typename adjlist_t::iterator arc_iterator;

  /** The arc type.  */
  typedef typename adjlist_t::arc_t arc_t;

  /** The distance type.  */
  typedef typename node_t::distance_t distance_t;

 protected:
  /** Given a graph, initializes the source node of the graph a the
      source, and resets the scan counter to zero.  */
  shortest_path_algorithm(graph_t& graph) : g(graph), scans(0) {
    g.source().set_source();
  }

 private:
  /** Counts the number of scans.  */
  unsigned scans;
 public:
  /** Return the number of scans.  */
  unsigned scan_count() const { return scans; }

  /** Incrementas the scan counte.  */
  void scan_tick() { ++scans; }

 protected:
  /** The graph we're working on.  */
  graph_t& g;
 public:
  /** Returns a reference to the graph.  */
  graph_t& get_graph() { return g; }

  /* Return the next node to be scanned, or NULL if the algorithm
     terminated.  This must be specialized in subclasses, but it is
     not subject to virtual dispatching.  */
  node_t *select() {
    return 0;
  }

  /* Return true iff the node was relabeled.  */
  bool test_relabel(arc_t& arc) {
    return arc.to().test_relabel(arc);
  }

  /* Mark the node as scanned.  */
  void scanned(node_t& node) {
    return node.mark_scanned();
  }

  /** Run the algorithm, i.e., until select() returns NULL,
      test_relabel all arcs of the node returned by select(), then
      tick the scan counter and mark the node as scanned.  */
  void run() {
    algorithm_t &algo = *static_cast<algorithm_t*>(this); // downcast it
    while(node_t *node_ptr = algo.select()) { // until node_ptr is NULL
      node_t &from = *node_ptr;
      adjlist_t& adjlist = from.adjlist(); // get its adjacency list
      for(arc_iterator arc_i = adjlist.begin(), arc_end = adjlist.end();
	  arc_i != arc_end; ++arc_i) // iterate on its arcs
	algo.test_relabel(*arc_i); // relabeling its children when possible
      algo.scan_tick(); // tick the scan counter
      algo.scanned(from); // and mark the node as scanned
    }
  }

  /** Output a summary of the algorithm.  */
  void summarize(std::ostream& out) {
    out << "Problem title: " << g.get_title() << endl;
    out << "Nodes: " << g.size() << endl;
    out << "Arcs: " << g.arc_count() << endl;
    out << "Sum of distances: " << g.sum_dist() << endl;
    out << "Number of scans: " << scan_count() << endl;
  }
};

#endif
